bl_info = {
    "name": "Ragdoll Tools",
    "category": "Rigging",
    "version": (1, 2),
    "blender": (2, 7, 9),
    "location": "Spacebar->Ragdoll tools pie menu",
    "description": "Tools for generating ragdolls vfxmed.com ",
    "wiki_url": "https://github.com/xbodya13/ragdoll_tools_doc",
    "tracker_url": "https://github.com/xbodya13/ragdoll_tools_doc/issues"
}

import bpy
import mathutils
# import os
import random
import math
import bmesh
import inspect

ragdoll_name = "ragdoll"

bone_name = "bone"
hit_box_name = "hit_box"

location_const_name = "location_const_with_long_name"
rotation_const_name = "rotation_const_with_long_name"
scale_const_name = "scale_const_with_long_name"

const_names = {'COPY_LOCATION': location_const_name, 'COPY_ROTATION': rotation_const_name,
               'COPY_SCALE': scale_const_name}
follow_bone_name = "follow_bone"

helpers_group_name = "ragdoll_helpers_group"
hit_box_group_name = "hit_box_group"

default_rbc_settings = {
    "type": {"default": 'GENERIC'},
    "enabled": {"default": True},
    "disable_collisions": {"default": True},

    "use_limit_lin_x": {"default": True},
    "use_limit_lin_y": {"default": True},
    "use_limit_lin_z": {"default": True},

    "use_limit_ang_x": {"default": True},
    "use_limit_ang_y": {"default": True},
    "use_limit_ang_z": {"default": True},

    "limit_lin_x_lower": {"default": 0.0},
    "limit_lin_y_lower": {"default": 0.0},
    "limit_lin_z_lower": {"default": 0.0},

    "limit_lin_x_upper": {"default": 0.0},
    "limit_lin_y_upper": {"default": 0.0},
    "limit_lin_z_upper": {"default": 0.0},

    "limit_ang_x_lower": {"default": -0.5235987901687622},
    "limit_ang_y_lower": {"default": -0.5235987901687622},
    "limit_ang_z_lower": {"default": -0.5235987901687622},

    "limit_ang_x_upper": {"default": 0.5235987901687622},
    "limit_ang_y_upper": {"default": 0.5235987901687622},
    "limit_ang_z_upper": {"default": 0.5235987901687622},

}

default_rb_settings = {
    "type": {"default": 'ACTIVE'},
    "enabled": {"default": True},
    "collision_groups": {
        "default": [True, False, False, False, False, False, False, False, False, False, False, False, False, False,
                    False, False, False, False, False, False]},
    "use_deactivation": {"default": False},
    "collision_shape": {"default": 'CONVEX_HULL'},
    "mesh_source": {"default": 'DEFORM'},
}


def make_rigid(to_rigid, context, settings=None):
    if len(to_rigid) != 0:
        over_context = context.copy()
        over_context["object"] = to_rigid[0]
        over_context["selected_objects"] = to_rigid
        bpy.ops.rigidbody.objects_add(over_context, type='ACTIVE')
        if settings != None:
            for item in to_rigid:
                copy_settings(settings, item.rigid_body, settings.property_keys)


def add_const(hit_box, type, mute, with_driver, target, bone=None):
    if bone != None:
        out_const = bone.constraints.new(type)
        out_const.name = const_names[type]
    else:
        out_const = hit_box.constraints.new(type)
        out_const.name = const_names[type]
    out_const.mute = mute
    out_const.target = target

    if with_driver:
        const_driver = out_const.driver_add("mute").driver
        const_driver.type = 'SCRIPTED'

        if bone != None:
            const_driver.expression = follow_bone_name
        else:
            const_driver.expression = "not " + follow_bone_name

        follow_bone = const_driver.variables.new()
        follow_bone.type = "SINGLE_PROP"
        follow_bone.name = follow_bone_name
        follow_bone.targets[0].id_type = 'OBJECT'
        follow_bone.targets[0].id = hit_box
        follow_bone.targets[0].data_path = follow_bone_name


def link_hit_box_and_bone(hit_box, pose_bone, empty_draw_type, empty_draw_size, group_name, context):
    hit_box.follow_bone = False

    hit_box_empty = bpy.data.objects.new(hit_box.name + "_empty", None)
    hit_box_empty.empty_draw_type = empty_draw_type
    hit_box_empty.empty_draw_size = empty_draw_size
    hit_box_empty.matrix_world = hit_box.matrix_world

    hit_box_empty.parent_to(pose_bone)

    context.scene.objects.link(hit_box_empty)

    bone_empty = bpy.data.objects.new(pose_bone.id_data.name + "_" + pose_bone.name + "_empty", None)
    bone_empty.empty_draw_type = empty_draw_type
    bone_empty.empty_draw_size = empty_draw_size
    bone_empty.matrix_world = pose_bone.matrix_world

    bone_empty.parent_to(hit_box)

    context.scene.objects.link(bone_empty)

    if group_name != None:
        if group_name in bpy.data.groups:
            group = bpy.data.groups[group_name]
        else:
            group = bpy.data.groups.new(group_name)
        group.objects.link(bone_empty)
        group.objects.link(hit_box_empty)

    add_const(hit_box, type='COPY_LOCATION', mute=True, with_driver=True, target=hit_box_empty)
    add_const(hit_box, type='COPY_ROTATION', mute=True, with_driver=True, target=hit_box_empty)
    add_const(hit_box, type='COPY_SCALE', mute=False, with_driver=False, target=hit_box_empty)

    add_const(hit_box, type='COPY_LOCATION', mute=False, with_driver=True, target=bone_empty, bone=pose_bone)
    add_const(hit_box, type='COPY_ROTATION', mute=False, with_driver=True, target=bone_empty, bone=pose_bone)


def get_hit_bone(hit_box):
    out_set = set()
    try:
        for const_name in (location_const_name, rotation_const_name, scale_const_name):
            target = hit_box.constraints[const_name].target
            out_set.add(target.parent.pose.bones[target.parent_bone])
    except:
        pass

    if len(out_set) == 1:
        return out_set.pop()
    else:
        return None


def get_hit_box(hit_bone):
    out_set = set()
    try:
        for const_name in (location_const_name, rotation_const_name):
            target = hit_bone.constraints[const_name].target
            out_set.add(target.parent)
    except:
        pass

    if len(out_set) == 1:
        return out_set.pop()
    else:
        return None


def get_pair(item):
    hit_box = None
    hit_bone = None
    if is_mesh(item):
        hit_bone = get_hit_bone(item)
        hit_box = get_hit_box(hit_bone)
        if hit_box != item:
            hit_box = None

    if type(item) == bpy.types.PoseBone:
        hit_box = get_hit_box(item)
        hit_bone = get_hit_bone(hit_box)
        if hit_bone != item:
            hit_bone = None

    return hit_box, hit_bone


def random_matrix():
    return mathutils.Matrix.Translation([random.randint(0, 5), random.randint(0, 5), random.randint(0, 5)])

def compose_matrix(l=mathutils.Vector((0,0,0)),r=mathutils.Quaternion((1,0,0,0)),s=mathutils.Vector((1,1,1))):
    print(s)
    # return mathutils.Matrix.Translation(l)*mathutils.Matrix.Rotation(r.angle,4,r.axis)*mathutils.Matrix.Scale(s[0],4,[0,1,0])
    # return  mathutils.Matrix.Identity(4) *mathutils.Matrix.Scale(s.length, 4)
    scale_matrix=mathutils.Matrix()
    scale_matrix[0][0] = s[0]
    scale_matrix[1][1] = s[1]
    scale_matrix[2][2] = s[2]
    return mathutils.Matrix.Translation(l) * mathutils.Matrix.Rotation(r.angle, 4, r.axis)*scale_matrix

def get_center_of_mass(mesh):
    # print(sum([v.co for v in mesh.vertices],mathutils.Vector((0,0,0))))
    out = mathutils.Vector((0, 0, 0))
    if len(mesh.vertices)!=0:
        out=sum([ v.co for v  in mesh.vertices],out)/len(mesh.vertices)
    return out

def set_origin(some_object,matrix):
    shift_matrix=some_object.matrix_world.inverted()*matrix
    some_object.matrix_world*=shift_matrix
    some_object.data.transform(shift_matrix.inverted())

    # some_object.matrix_world=mathutils.Matrix.Identity(4)




class BakeToBone(bpy.types.Operator):
    """Bake connected hitbox or bones animation to keyframes """
    bl_idname = "ragdoll.bake_to_bone"
    bl_label = "Bake"
    bl_options = {'REGISTER', 'UNDO'}

    bake_mode = bpy.props.EnumProperty(name="Bake mode",
                                       items=[
                                           ('BONES', "Bones animation bake", "Bakes bones animation to keyframes",
                                            'NONE', 0),
                                           ('BOXES', "Hitboxes animation bake", "Bakes hitbox animation to keyframes",
                                            'NONE', 1),
                                           ('BOTH', "Both", "Bakes hitbox and bones animation to keyframes", 'NONE', 2)
                                       ]
                                       )
    use_selected_bones_only = bpy.props.BoolProperty(name="Use selected bones only")
    frame_start = bpy.props.IntProperty(
        name="Start Frame",
        description="Start frame for baking",
        min=0, max=300000,
        default=1,
    )
    frame_end = bpy.props.IntProperty(
        name="End Frame",
        description="End frame for baking",
        min=1, max=300000,
        default=250,
    )

    def invoke(self, context, event):

        return context.window_manager.invoke_props_dialog(self)

    def execute(self, context):

        frame_range = range(self.frame_start, self.frame_end)

        hit_bone_set = set()
        hit_box_set = set()

        def add_to_set(item):
            hit_box, hit_bone = get_pair(item)
            if None not in (hit_box, hit_bone):
                hit_box_set.add(hit_box)
                hit_bone_set.add(hit_bone)

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                selected_pose_bones = context.selected_pose_bones
            else:
                selected_pose_bones = context.visible_pose_bones
            for pose_bone in selected_pose_bones:
                add_to_set(pose_bone)


        else:
            for selected_object in context.selected_objects:
                if context.mode != 'POSE':
                    if is_armature(selected_object):
                        for pose_bone in selected_object.pose.bones:
                            if not pose_bone.bone.hide:
                                if pose_bone.bone.select or not self.use_selected_bones_only:
                                    if any([bl and al for bl, al in
                                            zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                        add_to_set(pose_bone)

                if is_mesh(selected_object):
                    add_to_set(selected_object)

        if self.bake_mode == 'BONES':
            bake_items = hit_bone_set
        if self.bake_mode == 'BOXES':
            bake_items = hit_box_set
        if self.bake_mode == 'BOTH':
            bake_items = hit_box_set.union(hit_bone_set)



        if len(bake_items) != 0:
            last_frame = context.scene.frame_current

            trajectories = [[] for item in bake_items]
            for f in frame_range:
                # context.scene.frame_set(f)
                # context.scene.frame_set(f)
                context.scene.frame_set(f)
                context.scene.frame_set(f)
                # context.scene.update()
                # context.scene.frame_current=f

                for trajectory, item in zip(trajectories, bake_items):
                    if type(item) == bpy.types.PoseBone:
                        # item.matrix=item.matrix*mathutils.Matrix.Identity(4)
                        trajectory.append((item.id_data.convert_space(item, item.matrix, 'POSE', 'LOCAL'), f))
                    if type(item) == bpy.types.Object:
                        # item.matrix_world = item.matrix_world * mathutils.Matrix.Identity(4)
                        trajectory.append((item.matrix_world.copy(), f))

            options = {'INSERTKEY_NEEDED'}
            # options=set()
            # options = {'INSERTKEY_VISUAL'}

            for item, trajectory in zip(bake_items, trajectories):
                euler_prev = None
                # print(trajectory[0])
                for matrix, f in trajectory:
                    item.matrix_basis = matrix.copy()

                    item.keyframe_insert("location", -1, f, item.name, options)

                    rotation_mode = item.rotation_mode
                    if rotation_mode == 'QUATERNION':
                        item.keyframe_insert("rotation_quaternion", -1, f, item.name, options)
                    elif rotation_mode == 'AXIS_ANGLE':
                        item.keyframe_insert("rotation_axis_angle", -1, f, item.name, options)
                    else:  # euler, XYZ, ZXY etc
                        if euler_prev is not None:
                            euler = item.rotation_euler.copy()
                            euler.make_compatible(euler_prev)
                            item.rotation_euler = euler
                            euler_prev = euler
                            del euler
                        else:
                            euler_prev = item.rotation_euler.copy()
                        item.keyframe_insert("rotation_euler", -1, f, item.name, options)

                    item.keyframe_insert("scale", -1, f, item.name, options)

            context.scene.frame_set(last_frame)
            context.scene.frame_set(last_frame)

        return {'FINISHED'}


def register_extensions():
    bpy.types.PHYSICS_PT_rigid_body.append(draw_follow)

    setattr(bpy.types.Object, follow_bone_name, bpy.props.BoolProperty(name="Follow bone", default=True))

    bpy.types.Object.parent_to = make_parent
    bpy.types.Object.location_world = property(object_location_world)
    bpy.types.Object.center_world = property(object_location_world)

    bpy.types.PoseBone.matrix_world = property(pose_bone_world_matrix)
    bpy.types.PoseBone.matrix_world_rest = property(pose_bone_world_rest_matrix)

    bpy.types.PoseBone.head_world = property(pose_bone_head_world)
    bpy.types.PoseBone.tail_world = property(pose_bone_tail_world)
    bpy.types.PoseBone.center_world = property(pose_bone_center_world)

    bpy.types.PoseBone.head_world_rest = property(pose_bone_head_world_rest)
    bpy.types.PoseBone.tail_world_rest = property(pose_bone_tail_world_rest)
    bpy.types.PoseBone.center_world_rest = property(pose_bone_center_world_rest)


def draw_follow(self, context):
    if context.object != None:
        if None not in get_pair(context.object):
            layout = self.layout
            layout.label(text="Ragdoll:")
            layout.prop(context.object, "follow_bone")


def make_parent(child_item, parent_item, use_rest=False):
    if type(parent_item) == bpy.types.PoseBone:
        child_item.parent = parent_item.id_data
        child_item.parent_type = 'BONE'
        child_item.parent_bone = parent_item.name
        if use_rest:
            child_item.matrix_parent_inverse = (
                        parent_item.id_data.matrix_world * parent_item.bone.matrix_local * mathutils.Matrix.Translation(
                    mathutils.Vector([0, parent_item.bone.length, 0]))).inverted()
        else:
            child_item.matrix_parent_inverse = (
                        parent_item.id_data.matrix_world * parent_item.matrix * mathutils.Matrix.Translation(
                    mathutils.Vector([0, parent_item.bone.length, 0]))).inverted()
    if type(parent_item) == bpy.types.Object:
        child_item.parent = parent_item
        child_item.matrix_parent_inverse = parent_item.matrix_world.inverted()


def object_location_world(some_object):
    return some_object.matrix_world.to_translation()


def pose_bone_world_matrix(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.matrix


def pose_bone_world_rest_matrix(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.bone.matrix_local


def pose_bone_head_world(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.head


def pose_bone_tail_world(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.tail


def pose_bone_center_world(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.center


def pose_bone_head_world_rest(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.bone.head_local


def pose_bone_tail_world_rest(pose_bone):
    return pose_bone.id_data.matrix_world * pose_bone.bone.tail_local


def pose_bone_center_world_rest(pose_bone):
    return (
                       pose_bone.id_data.matrix_world * pose_bone.bone.head_local + pose_bone.id_data.matrix_world * pose_bone.bone.tail_local) * 0.5


def print_context(context):
    work_context = context.copy()
    print("\n\n|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||\n\n")
    for key in sorted(work_context.keys()):
        print("->", key, "   ", work_context[key])


def print_dir(item):
    print("\n\n|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||\n\n")
    for key in sorted(dir(item)):
        print("->", key, "   ", str(getattr(item, key)))


def prop_from_type(rna_type, **override):
    out = None

    if type(rna_type) == bpy.types.EnumProperty:
        out = bpy.props.EnumProperty(
            items=[(item.identifier, item.name, item.description, item.icon, x) for x, item in
                   enumerate(rna_type.enum_items)],
            name=rna_type.name,
            description=rna_type.description,
            default=rna_type.default
        )
    if type(rna_type) == bpy.types.BoolProperty:
        if rna_type.is_array:
            subtype_mapping = {'LAYER_MEMBERSHIP': 'LAYER'}

            out = bpy.props.BoolVectorProperty(
                name=rna_type.name,
                description=rna_type.description,
                subtype=subtype_mapping[rna_type.subtype],
                size=rna_type.array_length,
                default=rna_type.default_array

            )

        else:
            out = bpy.props.BoolProperty(
                name=rna_type.name,
                description=rna_type.description,
                default=rna_type.default,
                subtype=rna_type.subtype
            )
    if type(rna_type) == bpy.types.FloatProperty:
        out = bpy.props.FloatProperty(
            name=rna_type.name,
            description=rna_type.description,
            default=rna_type.default,
            subtype=rna_type.subtype,

            min=rna_type.hard_min,
            max=rna_type.hard_max,
            soft_min=rna_type.soft_min,
            soft_max=rna_type.soft_max,

            step=rna_type.step,
            precision=rna_type.precision

        )
    if type(rna_type) == bpy.types.IntProperty:
        out = bpy.props.IntProperty(
            name=rna_type.name,
            description=rna_type.description,
            default=rna_type.default,
            subtype=rna_type.subtype,

            min=rna_type.hard_min,
            max=rna_type.hard_max,
            soft_min=rna_type.soft_min,
            soft_max=rna_type.soft_max,

            step=rna_type.step,

        )
    for key, value in override.items():
        if key in out[1].keys():
            out[1][key] = value
    return out


def pgroup_from_type(b_type, **override):
    def pgroup_from_type_inner(c_object):
        property_keys = set()
        for p_name, p_type in b_type.bl_rna.properties.items():
            if prop_from_type(p_type) != None:
                if p_name in override.keys():
                    setattr(c_object, p_name, prop_from_type(p_type, **override[p_name]))
                else:
                    setattr(c_object, p_name, prop_from_type(p_type))
                property_keys.add(p_name)

        setattr(c_object, "property_keys", property_keys)
        return c_object

    return pgroup_from_type_inner


def inner_pgroups_to_pointers(c_object):
    pgroups = []
    for pgroup_name in vars(c_object):
        tmp = getattr(c_object, pgroup_name)
        if inspect.isclass(tmp):
            if issubclass(tmp, bpy.types.PropertyGroup):
                pgroups.append(pgroup_name)
    for pgroup_name in pgroups:
        tmp = getattr(c_object, pgroup_name)
        d_name = pgroup_name + "_definition"
        setattr(c_object, d_name, tmp)
        setattr(c_object, pgroup_name, bpy.props.PointerProperty(type=getattr(c_object, d_name)))

    return c_object


def is_armature(t_object):
    return t_object != None and type(t_object) == bpy.types.Object and type(t_object.data) == bpy.types.Armature


def is_mesh(t_object):
    return t_object != None and type(t_object) == bpy.types.Object and type(t_object.data) == bpy.types.Mesh


def copy_settings(from_class, to_class, keys):
    for key in keys:
        if hasattr(from_class, key) and hasattr(to_class, key):
            setattr(to_class, key, getattr(from_class, key))


def world_matrix(some_object):
    out_matrix = mathutils.Matrix.Identity(4)
    if type(some_object) == bpy.types.Bone:
        out_matrix = some_object.id_data.matrix_world * some_object.matrix_local
    if type(some_object) == bpy.types.PoseBone:
        out_matrix = some_object.id_data.matrix_world * some_object.matrix
    if type(some_object) == bpy.types.Object:
        out_matrix = some_object.matrix_world
    return out_matrix


def world_distance(object_a, object_b):
    return (world_matrix(object_a).to_translation() - world_matrix(object_b).to_translation()).length


def length(a, b):
    return (a - b).length


def get_nearest(some_object, targets):
    min_l = 0
    min_object = None
    for target in targets:

        l = length(some_object.center_world, target.center_world)
        if min_object == None or l < min_l:
            min_l = l
            min_object = target
    return min_object


def clear_parent(item):
    pass


@inner_pgroups_to_pointers
class LinkHitBoxAndBone(bpy.types.Operator):
    """Connect selected mesh objects to nearest bones in selected armatures"""
    bl_idname = "ragdoll.link_hit_boxes_and_bones"
    bl_label = "Connect"
    bl_options = {'REGISTER', 'UNDO'}

    use_selected_bones_only = bpy.props.BoolProperty(name="Use selected bones only")

    class rig_settings(bpy.types.PropertyGroup):
        empty_draw_type = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_type"], name="Display",
                                         default='CUBE')
        empty_draw_size = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_size"], default=0.2)
        helpers_group = bpy.props.StringProperty(name="Group", default=helpers_group_name)

    def check(self, context):
        return True

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "use_selected_bones_only")

        layout.prop(self.rig_settings, "helpers_group")
        layout.prop(self.rig_settings, "empty_draw_type")
        layout.prop(self.rig_settings, "empty_draw_size")

    def execute(self, context):
        mesh_list = []
        pose_bone_list = []

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                selected_pose_bones = context.selected_pose_bones
            else:
                selected_pose_bones = context.visible_pose_bones
            for pose_bone in selected_pose_bones:
                if None in get_pair(pose_bone):
                    pose_bone_list.append(pose_bone)

        for selected_object in context.selected_objects:
            if context.mode != 'POSE':
                if is_armature(selected_object):
                    for pose_bone in selected_object.pose.bones:
                        if not pose_bone.bone.hide:
                            if pose_bone.bone.select or not self.use_selected_bones_only:
                                if any([bl and al for bl, al in
                                        zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                    if None in get_pair(pose_bone):
                                        pose_bone_list.append(pose_bone)

            if is_mesh(selected_object):
                if None in get_pair(selected_object):
                    mesh_list.append(selected_object)

        remove_rig(pose_bone_list)
        remove_rig(mesh_list)


        if len(mesh_list) <= len(pose_bone_list):
            tmp_set = set(pose_bone_list)
            for mesh in mesh_list:
                nearest_pose_bone = get_nearest(mesh, tmp_set)
                link_hit_box_and_bone(mesh, nearest_pose_bone, self.rig_settings.empty_draw_type,
                                      self.rig_settings.empty_draw_size, self.rig_settings.helpers_group, context)
                tmp_set.remove(nearest_pose_bone)
        else:
            tmp_set = set(mesh_list)
            for pose_bone in pose_bone_list:
                nearest_mesh = get_nearest(pose_bone, tmp_set)
                link_hit_box_and_bone(nearest_mesh, pose_bone, self.rig_settings.empty_draw_type,
                                      self.rig_settings.empty_draw_size, self.rig_settings.helpers_group, context)
                tmp_set.remove(nearest_mesh)

        return {'FINISHED'}


def remove_rig(items):
    for item in items:
        constraints_to_delete = set()
        for constraint in item.constraints:
            if constraint.name in const_names.values():
                if type(item) == bpy.types.PoseBone:
                    item.id_data.driver_remove(
                        'pose.bones["{0}"].constraints["{1}"].mute'.format(item.name, constraint.name))
                if type(item) == bpy.types.Object:
                    item.driver_remove('constraints["{0}"].mute'.format(constraint.name))
                constraints_to_delete.add(constraint)
        for constraint in constraints_to_delete:
            item.constraints.remove(constraint)


class UnlinkHitBoxAndBone(bpy.types.Operator):
    """Disconnect bones from hitboxes"""
    bl_idname = "ragdoll.unlink_hit_boxes_and_bones"
    bl_label = "Disconnect"
    bl_options = {'REGISTER', 'UNDO'}

    use_selected_bones_only = bpy.props.BoolProperty(name="Use selected bones only")

    delete_helpers = bpy.props.BoolProperty(name="Delete helpers", description="Delete related helpers", default=True)
    delete_hit_boxes = bpy.props.BoolProperty(name="Delete hitboxes", description="Delete hitboxes", default=False)
    delete_constraints = bpy.props.BoolProperty(name="Delete related constraints", description="Delete related constraints", default=True)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "use_selected_bones_only")
        layout.prop(self, "delete_helpers")
        layout.prop(self, "delete_hit_boxes")
        layout.prop(self, "delete_constraints")

    def execute(self, context):

        hit_bone_set = set()
        hit_box_set = set()
        helpers_to_delete = set()

        unlinked_items = set()

        mesh_set=set()

        def add_to_set(item):
            hit_box, hit_bone = get_pair(item)
            if None not in (hit_box, hit_bone):
                hit_box_set.add(hit_box)
                hit_bone_set.add(hit_bone)
                if self.delete_helpers:
                    helpers_to_delete.add(hit_box.constraints[location_const_name].target)
                    helpers_to_delete.add(hit_bone.constraints[location_const_name].target)

            else:
                unlinked_items.add(item)
            if is_mesh(item):
                mesh_set.add(item)

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                selected_pose_bones = context.selected_pose_bones
            else:
                selected_pose_bones = context.visible_pose_bones
            for pose_bone in selected_pose_bones:
                add_to_set(pose_bone)


        else:
            for selected_object in context.selected_objects:
                if context.mode != 'POSE':
                    if is_armature(selected_object):
                        for pose_bone in selected_object.pose.bones:
                            if not pose_bone.bone.hide:
                                if pose_bone.bone.select or not self.use_selected_bones_only:
                                    if any([bl and al for bl, al in zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                        add_to_set(pose_bone)

                if is_mesh(selected_object):
                    add_to_set(selected_object)

        remove_rig(hit_box_set)
        remove_rig(hit_bone_set)
        remove_rig(unlinked_items)


        if self.delete_constraints:
            constraints_to_delete=set()
            if context.scene.rigidbody_world!=None:
                if context.scene.rigidbody_world.constraints!=None:
                    for item in context.scene.rigidbody_world.constraints.objects:
                        if item.rigid_body_constraint!=None:
                            if item.rigid_body_constraint.object1 in hit_box_set:
                                constraints_to_delete.add(item)
                            else:
                                if item.rigid_body_constraint.object2 in hit_box_set:
                                    constraints_to_delete.add(item)
            for constraint in constraints_to_delete:
                bpy.data.objects.remove(constraint, True)


        if self.delete_helpers:
            for helper in helpers_to_delete:
                bpy.data.objects.remove(helper, True)
        if self.delete_hit_boxes:
            for hit_box in hit_box_set:
                bpy.data.objects.remove(hit_box, True)




        bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP', iterations=1)

        return {'FINISHED'}


@inner_pgroups_to_pointers
class SetRBCSettings(bpy.types.Operator):
    """Set rigid body constraint settings on selected objects"""
    bl_idname = "ragdoll.set_rigid_body_constraint_settings"
    bl_label = "Set rigid body constraint settings"
    bl_options = {'REGISTER', 'UNDO'}

    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyConstraint, **default_rbc_settings)
        class rigid_body_constraint(bpy.types.PropertyGroup): pass

    empty_draw_type = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_type"], name="Display",
                                     default='CUBE')
    empty_draw_size = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_size"], default=0.2)

    def check(self, context):
        return True

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "empty_draw_type")
        layout.prop(self, "empty_draw_size")
        layout.separator()
        bpy.types.PHYSICS_PT_rigid_body_constraint.draw(self, self)

    def invoke(self, context, event):
        if context.object != None:
            if hasattr(context.object, "rigid_body_constraint"):
                copy_settings(context.object.rigid_body_constraint, self.object.rigid_body_constraint,
                              self.object.rigid_body_constraint.property_keys)
                if context.object.data == None:
                    self.empty_draw_type = context.object.empty_draw_type
                    self.empty_draw_size = context.object.empty_draw_size

        return self.execute(context)

    def execute(self, context):

        for item in context.selected_objects:
            if hasattr(item, "rigid_body_constraint"):
                copy_settings(self.object.rigid_body_constraint, item.rigid_body_constraint,
                              self.object.rigid_body_constraint.property_keys)
            if item.data == None:
                item.empty_draw_type = self.empty_draw_type
                item.empty_draw_size = self.empty_draw_size

        return {'FINISHED'}


@inner_pgroups_to_pointers
class SetRBSettings(bpy.types.Operator):
    """Set rigid body settings on selected objects"""
    bl_idname = "ragdoll.set_rigid_body_settings"
    bl_label = "Set rigid body settings"
    bl_options = {'REGISTER', 'UNDO'}

    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyObject)
        class rigid_body(bpy.types.PropertyGroup): pass

    def check(self, context):
        return True

    def draw(self, context):
        bpy.types.PHYSICS_PT_rigid_body.draw(self, self)
        bpy.types.PHYSICS_PT_rigid_body_collisions.draw(self, self)
        bpy.types.PHYSICS_PT_rigid_body_dynamics.draw(self, self)

    def invoke(self, context, event):
        if context.object != None:
            if hasattr(context.object, "rigid_body"):
                copy_settings(context.object.rigid_body, self.object.rigid_body, self.object.rigid_body.property_keys)

        return self.execute(context)

    def execute(self, context):

        for item in context.selected_objects:
            if hasattr(item, "rigid_body"):
                copy_settings(self.object.rigid_body, item.rigid_body, self.object.rigid_body.property_keys)

        return {'FINISHED'}


@inner_pgroups_to_pointers
class MakeRigidBodyConstraints(bpy.types.Operator):
    """Connect selected rigid bodies with rigid body constraints"""
    bl_idname = "ragdoll.make_rigid_body_constraints"
    bl_label = "Generate rigid body constraints"
    bl_options = {'REGISTER', 'UNDO'}

    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyConstraint, **default_rbc_settings)
        class rigid_body_constraint(bpy.types.PropertyGroup): pass

    class constraint_settings(bpy.types.PropertyGroup):
        connection_pattern = bpy.props.EnumProperty(name="Connection pattern", default='SELECTED_TO_ACTIVE',
                                                    items=[
                                                        ('SELECTED_TO_ACTIVE', "Selected to active",
                                                         "Connect selected objects to active with rigid body constraint",
                                                         'NONE', 0),
                                                        ('CHAIN_BY_DISTANCE', "Chain by distance",
                                                         "Connect objects as a chain based on distance,starting at the active object",
                                                         'NONE', 1),
                                                        ('ALL', "All to all",
                                                         "Connect every object with all the others",
                                                         'NONE', 2),
                                                    ]
                                                    )

        connection_pattern_with_bones = bpy.props.EnumProperty(name="Connection pattern", default='FROM_BONES',
                                                               items=[
                                                                   ('SELECTED_TO_ACTIVE', "Selected to active",
                                                                    "Connect selected objects to active with rigid body constraint",
                                                                    'NONE', 0),
                                                                   ('CHAIN_BY_DISTANCE', "Chain by distance",
                                                                    "Connect objects as a chain based on distance,starting at the active object",
                                                                    'NONE', 1),

                                                                   ('ALL', "All to all",
                                                                    "Connect every object with all others",
                                                                    'NONE', 2),

                                                                   ('FROM_BONES', "Use connected bones hierarchy",
                                                                    "Use hierarchy of connected bones to connect objects with rigid body constraints",
                                                                    'NONE', 3),
                                                               ]
                                                               )

        parent_to = bpy.props.EnumProperty(name="Parent to", default='PARENT',
                                           items=[
                                               ('CHILD', "Child", "Child", 'NONE', 0),
                                               ('PARENT', "Parent", "Parent", 'NONE', 1),
                                               ('NONE', "None", "None", 'NONE', 2),
                                           ]
                                           )

        constraint_location = bpy.props.EnumProperty(name="Location", default='CHILD',
                                                     items=[
                                                         ('CHILD', "Child", "Child", 'NONE', 0),
                                                         ('PARENT', "Parent", "Parent", 'NONE', 1),
                                                         ('CENTER', "Center", "Center", 'NONE', 2),
                                                     ]
                                                     )

        origins_items = [
            ('ORIGIN', "Object origin", "Object origin", 'NONE', 0),
            ('HEAD', "Head of connected bone", "Head of connected bone", 'NONE', 1),
            ('TAIL', "Tail of connected bone", "Tail of connected bone", 'NONE', 2),
        ]

        origin_a = bpy.props.EnumProperty(name="Origin", items=origins_items, default='HEAD')
        origin_b = bpy.props.EnumProperty(name="Origin", items=origins_items, default='HEAD')

        constraint_orientation = bpy.props.EnumProperty(name="Orientation", default='CHILD',
                                                        items=[
                                                            ('CHILD', "Child", "Child", 'NONE', 0),
                                                            ('PARENT', "Parent", "Parent", 'NONE', 1),
                                                        ]
                                                        )
        constraint_orientation_with_bones = bpy.props.EnumProperty(name="Orientation", default='CHILD_BONE',
                                                                   items=[
                                                                       ('CHILD', "Child", "Child", 'NONE', 0),
                                                                       ('PARENT', "Parent", "Parent", 'NONE', 1),
                                                                       ('CHILD_BONE', "Child connected bone",
                                                                        "Child connected bone", 'NONE', 2),
                                                                       ('PARENT_BONE', "Parent connected bone",
                                                                        "Parent connected bone", 'NONE', 3),
                                                                   ]
                                                                   )

        empty_draw_type = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_type"], name="Display",
                                         default='SPHERE')
        empty_draw_size = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_size"], default=0.2)

    @classmethod
    def poll(self, context):

        self.has_connected_bones = False
        self.linked_meshes = {}
        self.linked_bones = {}
        self.meshes = []

        if context.mode == 'POSE':
            for pose_bone in context.selected_pose_bones:
                hit_box, hit_bone = get_pair(pose_bone)
                if None not in (hit_box, hit_bone):
                    if hit_box.rigid_body != None:
                        self.linked_meshes[hit_box] = hit_bone
                        self.linked_bones[hit_bone] = hit_box
                        self.meshes.append(hit_box)
        else:
            for selected_object in context.selected_objects:
                # print(selected_object.rigidbody)
                if is_mesh(selected_object) :
                    if selected_object.rigid_body != None:
                        self.meshes.append(selected_object)
                hit_box, hit_bone = get_pair(selected_object)
                if None not in (hit_box, hit_bone):
                    self.linked_meshes[hit_box] = hit_bone
                    self.linked_bones[hit_bone] = hit_box

        if len(self.meshes) >= 2:
            if len(self.linked_meshes) != 0:
                self.has_connected_bones = True
            return True
        else:
            return False

    def get_transform(self, parent_object, child_object):
        location = None
        rotation = None

        if self.has_connected_bones:
            if self.constraint_settings.constraint_orientation_with_bones == 'PARENT_BONE' and parent_object in self.linked_meshes:
                rotation = self.linked_meshes[parent_object].matrix_world.to_quaternion()
            if self.constraint_settings.constraint_orientation_with_bones == 'CHILD_BONE' and child_object in self.linked_meshes:
                rotation = self.linked_meshes[child_object].matrix_world.to_quaternion()

            if self.constraint_settings.constraint_orientation_with_bones == 'PARENT':
                rotation = parent_object.matrix_world.to_quaternion()
            if self.constraint_settings.constraint_orientation_with_bones == 'CHILD':
                rotation = child_object.matrix_world.to_quaternion()
            pass
        else:
            if self.constraint_settings.constraint_orientation == 'PARENT':
                rotation = parent_object.matrix_world.to_quaternion()
            if self.constraint_settings.constraint_orientation == 'CHILD':
                rotation = child_object.matrix_world.to_quaternion()

        locations = [None, None]
        hit_boxes = (parent_object, child_object)
        if self.constraint_settings.constraint_location == 'CENTER':
            origins = (self.constraint_settings.origin_a, self.constraint_settings.origin_b)
        else:
            origins = (self.constraint_settings.origin_a, self.constraint_settings.origin_a)

        for x, (origin, hit_box) in enumerate(zip(origins, hit_boxes)):

            if origin == 'ORIGIN' or hit_box not in self.linked_meshes:
                locations[x] = hit_box.matrix_world.to_translation()

            else:
                if origin == 'HEAD':
                    locations[x] = self.linked_meshes[hit_box].head_world
                if origin == 'TAIL':
                    locations[x] = self.linked_meshes[hit_box].tail_world

        if self.constraint_settings.constraint_location == 'PARENT':
            location = locations[0]

        if self.constraint_settings.constraint_location == 'CHILD':
            location = locations[1]

        if self.constraint_settings.constraint_location == 'CENTER':
            location = (locations[0] + locations[1]) * 0.5

        out_matrix = mathutils.Matrix.Translation(location) * mathutils.Matrix.Rotation(rotation.angle, 4,
                                                                                        rotation.axis)

        return out_matrix

    def make_const(self, parent_object, child_object, context):

        const_empty=self.init_const.copy()
        const_empty.name="const_empty"

        context.scene.objects.link(const_empty)

        context.scene.rigidbody_world.constraints.objects.link(const_empty)

        const_empty.rigid_body_constraint.object1 = parent_object
        const_empty.rigid_body_constraint.object2 = child_object

        const_empty.matrix_world = self.get_transform(parent_object, child_object)
        if self.constraint_settings.parent_to == 'PARENT':
            const_empty.parent_to(parent_object)
        if self.constraint_settings.parent_to == 'CHILD':
            const_empty.parent_to(child_object)

    def execute(self, context):



        if context.scene.rigidbody_world == None:
            bpy.ops.rigidbody.world_add()
        last_active = context.object



        self.init_const = bpy.data.objects.new("", None)
        self.init_const.empty_draw_type = self.constraint_settings.empty_draw_type
        self.init_const.empty_draw_size = self.constraint_settings.empty_draw_size
        context.scene.objects.link(self.init_const)
        context.scene.objects.active =self.init_const
        bpy.ops.rigidbody.constraint_add()
        copy_settings(self.object.rigid_body_constraint, self.init_const.rigid_body_constraint,self.object.rigid_body_constraint.property_keys)




        if self.has_connected_bones:
            pattern = self.constraint_settings.connection_pattern_with_bones
        else:
            pattern = self.constraint_settings.connection_pattern

        if pattern == 'SELECTED_TO_ACTIVE':
            parent_object = last_active
            if is_mesh(parent_object) and parent_object.rigid_body != None:
                for child_object in self.meshes:
                    if child_object != parent_object:
                        self.make_const(parent_object, child_object, context)

        if pattern == 'CHAIN_BY_DISTANCE':
            if is_mesh(last_active) and context.object.rigid_body != None:
                parent_object = last_active
            else:
                parent_object = self.meshes[0]

            tmp_hit_boxes = set(self.meshes)
            tmp_hit_boxes.remove(parent_object)
            while len(tmp_hit_boxes) != 0:
                child_object = min(tmp_hit_boxes, key=lambda v: length(v.location_world, parent_object.location_world))
                self.make_const(parent_object, child_object, context)
                tmp_hit_boxes.remove(child_object)
                parent_object = child_object

        if pattern == 'FROM_BONES':
            for pose_bone in self.linked_bones:
                if pose_bone.parent in self.linked_bones:
                    parent_object = self.linked_bones[pose_bone.parent]
                    child_object = self.linked_bones[pose_bone]
                    self.make_const(parent_object, child_object, context)


        if pattern == 'ALL':
            for x in range(len(self.meshes)):
                for y in range(x+1,len(self.meshes)):
                    self.make_const(self.meshes[x], self.meshes[y], context)

        bpy.data.objects.remove(self.init_const, True)
        context.scene.objects.active = last_active

        return {'FINISHED'}

    # def invoke(self,context,event):
    #     print("INVOKE")
    #     return {'FINISHED'}

    def check(self, context):
        return True

    def draw(self, context):
        layout = self.layout

        if self.has_connected_bones:
            layout.prop(self.constraint_settings, "connection_pattern_with_bones")
        else:
            layout.prop(self.constraint_settings, "connection_pattern")
        layout.prop(self.constraint_settings, "parent_to")

        transform_split = layout.split()
        location_col = transform_split.column()
        rotation_col = transform_split.column()

        location_col.label(text="Location:")
        location_col.prop(self.constraint_settings, "constraint_location", text="")

        if self.has_connected_bones:
            if self.constraint_settings.constraint_location == 'CENTER':
                split = location_col.split()
                col = split.column()
                col.prop(self.constraint_settings, "origin_a", text="")
                col = split.column()
                col.prop(self.constraint_settings, "origin_b", text="")
            else:
                location_col.prop(self.constraint_settings, "origin_a", text="")

        rotation_col.label(text="Orientation:")
        if self.has_connected_bones:
            rotation_col.prop(self.constraint_settings, "constraint_orientation_with_bones", text="")
        else:
            rotation_col.prop(self.constraint_settings, "constraint_orientation", text="")

        layout.separator()

        layout.prop(self.constraint_settings, "empty_draw_type")
        layout.prop(self.constraint_settings, "empty_draw_size")
        layout.separator()

        bpy.types.PHYSICS_PT_rigid_body_constraint.draw(self, self)


@inner_pgroups_to_pointers
class GenerateHitBoxes(bpy.types.Operator):
    """Generate hitboxes from armatures"""
    bl_idname = "ragdoll.generate_hit_boxes"
    bl_label = "Generate hitboxes"
    bl_options = {'REGISTER', 'UNDO'}

    def get_hit_box_name(self,pose_bone):
        return pose_bone.id_data.name + "_" + pose_bone.name + "_hitbox"

    def make_hit_box(self, mesh, pose_bone, context):
        out_object = bpy.data.objects.new(self.get_hit_box_name(pose_bone), mesh)
        origin_shift = mathutils.Vector([0, (pose_bone.bone.center - pose_bone.bone.head).length, 0])
        out_object.matrix_world = pose_bone.matrix_world * mathutils.Matrix.Translation(origin_shift)
        out_object.data.transform(mathutils.Matrix.Translation(-origin_shift))
        context.scene.objects.link(out_object)
        return out_object

    def capsule_mesh(self, bone,matrix=None):
        if self.mesh_settings.use_envelope:
            if bone.parent != None and bone.use_connect:
                head_radius = bone.parent.tail_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            else:
                head_radius = bone.head_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            tail_radius = bone.tail_radius * self.mesh_settings.tail_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
        else:
            head_radius = self.mesh_settings.head_radius
            tail_radius = self.mesh_settings.tail_radius
        head_shift = bone.length * self.mesh_settings.head_shift
        tail_shift = bone.length * self.mesh_settings.tail_shift

        u_segments = self.mesh_settings.segments
        v_segments = self.mesh_settings.segments

        out_mesh = None
        if head_radius != 0 and tail_radius != 0 and u_segments >= 2 and v_segments >= 2:
            out_name = bone.id_data.name + "_" + bone.name + "_hitbox"
            out_mesh = bpy.data.meshes.new(out_name)

            eps = 0.0001

            bm = bmesh.new()
            bm.from_mesh(out_mesh)

            head_sphere = bmesh.ops.create_uvsphere(bm, u_segments=u_segments, v_segments=v_segments,
                                                    diameter=head_radius)
            to_delete = [v for v in head_sphere["verts"] if v.co.z > eps]
            bmesh.ops.translate(bm, verts=head_sphere["verts"], vec=mathutils.Vector([0, 0, head_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context=1)

            tail_sphere = bmesh.ops.create_uvsphere(bm, u_segments=u_segments, v_segments=v_segments,
                                                    diameter=tail_radius)
            to_delete = [v for v in tail_sphere["verts"] if v.co.z < -eps]
            bmesh.ops.translate(bm, verts=tail_sphere["verts"],
                                vec=mathutils.Vector([0, 0, bone.vector.length - tail_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context=1)

            boundary = [edge for edge in bm.edges if edge.is_boundary]
            bmesh.ops.bridge_loops(bm, edges=boundary)

            bmesh.ops.transform(bm, verts=bm.verts, matrix=mathutils.Matrix.Rotation(-math.pi / 2, 4, 'X'))

            if matrix!=None:
                bmesh.ops.transform(bm, verts=bm.verts, matrix=matrix)

            bm.to_mesh(out_mesh)

            bm.free()

        return out_mesh

    def box_mesh(self, bone,matrix=None):
        if self.mesh_settings.use_envelope:
            if bone.parent != None and bone.use_connect:
                head_radius = bone.parent.tail_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            else:
                head_radius = bone.head_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            tail_radius = bone.tail_radius * self.mesh_settings.tail_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
        else:
            head_radius = self.mesh_settings.head_radius
            tail_radius = self.mesh_settings.tail_radius
        head_shift = bone.length * self.mesh_settings.head_shift
        tail_shift = bone.length * self.mesh_settings.tail_shift

        u_segments = self.mesh_settings.segments
        v_segments = self.mesh_settings.segments

        out_mesh = None
        if head_radius != 0 and tail_radius != 0 and u_segments >= 2 and v_segments >= 2:
            out_name = bone.id_data.name + "_" + bone.name + "_hitbox"
            out_mesh = bpy.data.meshes.new(out_name)

            eps = 0.0001

            bm = bmesh.new()
            bm.from_mesh(out_mesh)

            head_sphere = bmesh.ops.create_grid(bm, size=head_radius)
            bmesh.ops.reverse_faces(bm, faces=bm.faces)
            to_delete = [v for v in head_sphere["verts"] if v.co.z > eps]
            bmesh.ops.translate(bm, verts=head_sphere["verts"], vec=mathutils.Vector([0, 0, head_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context=1)

            tail_sphere = bmesh.ops.create_grid(bm, size=tail_radius)
            to_delete = [v for v in tail_sphere["verts"] if v.co.z < -eps]
            bmesh.ops.translate(bm, verts=tail_sphere["verts"],
                                vec=mathutils.Vector([0, 0, bone.vector.length - tail_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context=1)

            boundary = [edge for edge in bm.edges if edge.is_boundary]
            bmesh.ops.bridge_loops(bm, edges=boundary)

            bmesh.ops.transform(bm, verts=bm.verts, matrix=mathutils.Matrix.Rotation(-math.pi / 2, 4, 'X'))

            if matrix!=None:
                bmesh.ops.transform(bm, verts=bm.verts, matrix=matrix)

            bm.to_mesh(out_mesh)

            bm.free()

        return out_mesh

    def make_convex(self,mesh):
        if len(mesh.vertices)>=3:
            bm = bmesh.new()
            bm.from_mesh(mesh)
            out=bmesh.ops.convex_hull(bm,input=bm.verts)
            bmesh.ops.delete(bm, geom=out["geom_interior"], context=1)
            # bmesh.ops.delete(bm, geom=out["geom_unused"], context=1)
            # bmesh.ops.delete(bm, geom=out["geom_holes"], context=1)
            bm.to_mesh(mesh)
            bm.free()
        return mesh

    def recalculate_normals(self,mesh):
        bm = bmesh.new()
        bm.from_mesh(mesh)
        bmesh.ops.recalc_face_normals(bm, faces=bm.faces)
        bm.to_mesh(mesh)
        bm.free()
        return mesh

    def clear_data(self,some_object):


        # last_matrix=some_object.matrix_world.copy()
        last_matrix=compose_matrix(*some_object.matrix_world.decompose())


        some_object.data.materials.clear()
        some_object.vertex_groups.clear()
        some_object.constraints.clear()
        some_object.modifiers.clear()
        some_object.animation_data_clear()


        some_object.parent=None


        for vc in some_object.data.vertex_colors:
            some_object.data.vertex_colors.remove(vc)
        for uv in some_object.data.uv_textures:
            some_object.data.uv_textures.remove(uv)

        some_object.data.transform(last_matrix)
        some_object.matrix_world=mathutils.Matrix.Identity(4)

        # self.recalculate_normals(some_object.data)
        some_object.data.update()






    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyObject, **default_rb_settings)
        class rigid_body(bpy.types.PropertyGroup): pass

    settings_tab = bpy.props.EnumProperty(name="Settings",
                                          items=[
                                              ('GENERAL', "General", "Settings of generated hitboxes ", 'MESH_DATA', 0),
                                              ('RIG', "Connect", "Connection settings ", 'LOCKVIEW_ON', 1),
                                              ('RIGID_BODY', "Rigid body", "Rigid body settings of generated hitboxes",
                                               'MESH_ICOSPHERE', 3)
                                          ]
                                          )

    use_selected_bones_only = bpy.props.BoolProperty(name="Use selected bones only")

    generate_from = bpy.props.EnumProperty(name="Generate from",
                                           items=[
                                               ('BONES', "Bones","Generate hitboxes from bones", 'NONE', 0),
                                               ('MESH', "Mesh", "Generate hitboxes from vertex groups of the selected mesh", 'NONE', 1),
                                           ]
                                           )
    hit_box_group = bpy.props.StringProperty(name="Hitbox group", default=hit_box_group_name,
                                             description="Group name of generated hitboxes")

    class mesh_settings(bpy.types.PropertyGroup):
        radius = bpy.props.FloatProperty(name="Radius", default=0.1)
        segments = bpy.props.IntProperty(name="Segments", default=10)

        u_segments = bpy.props.IntProperty()
        v_segments = bpy.props.IntProperty()

        use_envelope = bpy.props.BoolProperty(name="Use bone envelope")

        head_radius = bpy.props.FloatProperty(name="Head radius", min=0, default=0.2)
        tail_radius = bpy.props.FloatProperty(name="Tail radius", min=0, default=0.2)

        envelope_distance_mlt = bpy.props.FloatProperty(name="Distance multiplier", min=0, default=0)
        head_radius_mlt = bpy.props.FloatProperty(name="Head radius multiplier", min=0, default=1)
        tail_radius_mlt = bpy.props.FloatProperty(name="Tail radius multiplier", min=0, default=1)

        head_shift = bpy.props.FloatProperty(name="Head shift", default=0.)
        tail_shift = bpy.props.FloatProperty(name="Tail shift", default=0.)

        scale=bpy.props.FloatVectorProperty(name="Scale",default=(1.,1.,1.),)


        bone_shape= bpy.props.EnumProperty(name="Hitbox shape",
                                           items=[
                                               ('CAPSULE', "Capsule", "Create hitboxes as capsules", 'NONE', 0),
                                               ('BOX', "Box", "Create hitboxes as boxes", 'NONE', 1),
                                           ]
                                           )

        mesh_shape=bpy.props.EnumProperty(name="Mesh type",
                                          items=[
                                              ('PARTS', "Mesh parts", "Shape from vertex groups (slow)", 'NONE', 0),
                                              ('CONVEX', "Convex", "Convex shapes from vertex groups(very slow)", 'NONE', 1),
                                              ('FANCY', "Fancy", "Smooth shape from vertex groups (slowest) ",'NONE', 2)
                                          ]
                                          )
        use_normalize=bpy.props.BoolProperty(name="Normalize",description="Normalize vertex group weights before processing",default=True)
        weight_threshold=bpy.props.FloatProperty(name="Threshold",description="Result mesh will only include verticies with a weight above this threshold", min=0, default=0.5,max=1.0)
        resolution=bpy.props.IntProperty(name="Mesh resolution", min=0, default=4)



    class rig_settings(bpy.types.PropertyGroup):
        create_rig = bpy.props.BoolProperty(name="Connect hitboxes to bones", default=True)
        empty_draw_type = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_type"], default='CUBE')
        empty_draw_size = prop_from_type(bpy.types.Object.bl_rna.properties["empty_draw_size"], default=0.2)
        helpers_group = bpy.props.StringProperty(name="Group", default=helpers_group_name,
                                                 description="Group name of helper empties")

    show_rb_settings = bpy.props.BoolProperty(name="Show rigid body settings")

    def check(self, context):
        return True

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "settings_tab", expand=True)
        layout.separator()

        if self.settings_tab == 'GENERAL':
            layout.prop(self, "generate_from")
            layout.prop(self, "hit_box_group")
            layout.prop(self, "use_selected_bones_only")
            # layout.separator()
            # layout.label("Mesh settings:")
            layout.separator()

            if self.generate_from=='BONES':

                layout.prop(self.mesh_settings, "bone_shape")


                if self.mesh_settings.bone_shape == 'CAPSULE':
                    layout.prop(self.mesh_settings, "segments")

                sub = layout.column(align=True)
                sub.prop(self.mesh_settings, "head_shift")
                sub.prop(self.mesh_settings, "tail_shift")
                sub.label("Scale:")
                sub.row().prop(self.mesh_settings, "scale",text="")
                layout.prop(self.mesh_settings, "use_envelope")
                sub = layout.column(align=True)
                if self.mesh_settings.use_envelope:
                    sub.prop(self.mesh_settings, "head_radius_mlt")
                    sub.prop(self.mesh_settings, "tail_radius_mlt")
                    layout.prop(self.mesh_settings, "envelope_distance_mlt")
                else:
                    sub.prop(self.mesh_settings, "head_radius")
                    sub.prop(self.mesh_settings, "tail_radius")

            if self.generate_from== 'MESH':
                layout.prop(self.mesh_settings, "mesh_shape")
                layout.prop(self.mesh_settings, "use_normalize")
                layout.prop(self.mesh_settings, "weight_threshold")
                if self.mesh_settings.mesh_shape== 'FANCY':
                    layout.prop(self.mesh_settings, "resolution")


        if self.settings_tab == 'RIG':
            layout.prop(self.rig_settings, "create_rig")
            if self.rig_settings.create_rig:
                layout.prop(self.rig_settings, "helpers_group")
                layout.prop(self.rig_settings, "empty_draw_type")
                layout.prop(self.rig_settings, "empty_draw_size")

        if self.settings_tab == 'RIGID_BODY':
            bpy.types.PHYSICS_PT_rigid_body.draw(self, self)
            bpy.types.PHYSICS_PT_rigid_body_collisions.draw(self, self)
            bpy.types.PHYSICS_PT_rigid_body_dynamics.draw(self, self)

    # def invoke(self, context, event):
    #     print("INVOKE")
    #     return {'FINISHED'}

    def execute(self, context):
        context.scene.update()

        pose_bone_list = []
        mesh_list=[]

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                pose_bone_list = [pose_bone for pose_bone in context.selected_pose_bones if None in get_pair(pose_bone)]
            else:
                pose_bone_list = [pose_bone for pose_bone in context.visible_pose_bones if None in get_pair(pose_bone)]

            if self.generate_from == 'MESH':
                mesh_list=[selected_object for selected_object in  context.selected_objects if is_mesh(selected_object) ]




        else:
            for selected_object in context.selected_objects:
                if is_armature(selected_object):
                    for pose_bone in selected_object.pose.bones:
                        if not pose_bone.bone.hide:
                            if pose_bone.bone.select or not self.use_selected_bones_only:
                                if any([bl and al for bl, al in
                                        zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                    if None in get_pair(pose_bone):
                                        pose_bone_list.append(pose_bone)
                if self.generate_from== 'MESH':
                    if is_mesh(selected_object):
                        mesh_list.append(selected_object)



        remove_rig(pose_bone_list)
        to_rigid = []



        if self.generate_from== 'MESH':
            if len(mesh_list)==1:
                source_object=mesh_list[0].copy()

                base_meshes=[]
                face_counters=[]

                meshed_pose_bones=[]


                if self.mesh_settings.use_normalize:
                    over_context=context.copy()
                    over_context["object"]=source_object
                    bpy.ops.object.vertex_group_normalize_all(over_context,group_select_mode='ALL', lock_active=False)
                for pose_bone in pose_bone_list:
                    if pose_bone.name in source_object.vertex_groups:
                        vertex_group=source_object.vertex_groups[pose_bone.name]
                        base_mesh=source_object.copy()



                        weight_edit = base_mesh.modifiers.new("", 'VERTEX_WEIGHT_EDIT')
                        weight_edit.use_remove = True
                        weight_edit.remove_threshold = self.mesh_settings.weight_threshold
                        weight_edit.vertex_group = vertex_group.name

                        mask = base_mesh.modifiers.new("", 'MASK')
                        mask.vertex_group = vertex_group.name

                        if self.mesh_settings.mesh_shape != 'FANCY':
                            decimate=base_mesh.modifiers.new("", 'DECIMATE')
                            face_counters.append(decimate)

                        context.scene.objects.link(base_mesh)

                        base_meshes.append(base_mesh)
                        meshed_pose_bones.append(pose_bone)



                if self.mesh_settings.mesh_shape== 'FANCY':
                    fancy_meshes = []
                    for base_mesh,pose_bone in zip(base_meshes,meshed_pose_bones):


                        fancy_mesh=bpy.data.objects.new(self.get_hit_box_name(pose_bone),self.make_convex(base_mesh.to_mesh(context.scene, True, 'RENDER', False, False)))
                        fancy_mesh.matrix_world=base_mesh.matrix_world.copy()
                        fancy_mesh.data.materials.clear()
                        context.scene.objects.link(fancy_mesh)

                        first_remesh = fancy_mesh.modifiers.new("", 'REMESH')
                        first_remesh.mode = 'SMOOTH'
                        first_remesh.octree_depth=self.mesh_settings.resolution

                        wrap = fancy_mesh.modifiers.new("", 'SHRINKWRAP')
                        wrap.target = base_mesh

                        second_remesh = fancy_mesh.modifiers.new("", 'REMESH')
                        second_remesh.mode = 'SMOOTH'
                        second_remesh.octree_depth = self.mesh_settings.resolution

                        decimate = fancy_mesh.modifiers.new("", 'DECIMATE')

                        fancy_meshes.append(fancy_mesh)
                        face_counters.append(decimate)

                    context.scene.update()

                    for fancy_mesh,face_counter,pose_bone in zip(fancy_meshes,face_counters,meshed_pose_bones):
                        if face_counter.face_count >= 1:

                            mesh_to_delete=fancy_mesh.data
                            fancy_mesh.data=fancy_mesh.to_mesh(context.scene, True, 'RENDER', False, False)
                            bpy.data.meshes.remove(mesh_to_delete, True)

                            fancy_mesh.modifiers.clear()

                            fancy_mesh.data.name=self.get_hit_box_name(pose_bone)
                            fancy_mesh.show_x_ray = True
                            set_origin(fancy_mesh,compose_matrix(fancy_mesh.matrix_world*get_center_of_mass(fancy_mesh.data),pose_bone.matrix_world.to_quaternion()))
                            fancy_mesh.data.update()
                            link_hit_box_and_bone(fancy_mesh, pose_bone, self.rig_settings.empty_draw_type,self.rig_settings.empty_draw_size, self.rig_settings.helpers_group,context)
                            to_rigid.append(fancy_mesh)
                        else:
                            bpy.data.meshes.remove(fancy_mesh.data, True)
                            bpy.data.objects.remove(fancy_mesh, True)

                    for base_mesh in base_meshes:
                        bpy.data.objects.remove(base_mesh, True)

                else:
                    context.scene.update()

                    for base_mesh, face_counter,pose_bone in zip(base_meshes, face_counters,meshed_pose_bones):
                        if face_counter.face_count >= 1:
                            if self.mesh_settings.mesh_shape == 'PARTS':
                                out_mesh = bpy.data.objects.new(self.get_hit_box_name(pose_bone),base_mesh.to_mesh(context.scene, True, 'RENDER', False,False))
                            if self.mesh_settings.mesh_shape == 'CONVEX':
                                out_mesh=bpy.data.objects.new(self.get_hit_box_name(pose_bone),self.make_convex(base_mesh.to_mesh(context.scene, True, 'RENDER', False, False)))

                            out_mesh.data.materials.clear()

                            out_mesh.data.name = self.get_hit_box_name(pose_bone)
                            out_mesh.show_x_ray = True
                            out_mesh.matrix_world=base_mesh.matrix_world.copy()
                            set_origin(out_mesh,compose_matrix(out_mesh.matrix_world*get_center_of_mass(out_mesh.data),pose_bone.matrix_world.to_quaternion()))
                            out_mesh.data.update()
                            link_hit_box_and_bone(out_mesh, pose_bone, self.rig_settings.empty_draw_type,self.rig_settings.empty_draw_size, self.rig_settings.helpers_group,context)
                            to_rigid.append(out_mesh)
                            context.scene.objects.link(out_mesh)


                        bpy.data.objects.remove(base_mesh, True)


                bpy.data.objects.remove(source_object, True)


        else:

            for pose_bone in pose_bone_list:
                mesh = None
                matrix=mathutils.Matrix.Scale(self.mesh_settings.scale[0],4,(1,0,0))*mathutils.Matrix.Scale(self.mesh_settings.scale[1],4,(0,1,0))*mathutils.Matrix.Scale(self.mesh_settings.scale[2],4,(0,0,1))
                if self.mesh_settings.bone_shape == 'CAPSULE':
                    mesh = self.capsule_mesh(pose_bone.bone,matrix)
                if self.mesh_settings.bone_shape == 'BOX':
                    mesh = self.box_mesh(pose_bone.bone,matrix)

                if mesh != None:
                    hit_box = self.make_hit_box(mesh, pose_bone, context)
                    to_rigid.append(hit_box)
                    if self.rig_settings.create_rig:
                        link_hit_box_and_bone(hit_box, pose_bone, self.rig_settings.empty_draw_type,self.rig_settings.empty_draw_size, self.rig_settings.helpers_group, context)
                        pass
                    # bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP', iterations=1)

        if len(to_rigid) != 0:
            if self.hit_box_group in bpy.data.groups:
                group = bpy.data.groups[self.hit_box_group]
            else:
                group = bpy.data.groups.new(self.hit_box_group)
            for item in to_rigid:
                group.objects.link(item)

        make_rigid(to_rigid, context, self.object.rigid_body)

        return {'FINISHED'}


class RagdollPieMenu(bpy.types.Menu):
    """I am help string"""
    bl_label = "Ragdoll pie menu"

    def draw(self, context):
        layout = self.layout
        pie = layout.menu_pie()

        pie.operator(GenerateHitBoxes.bl_idname, icon='MESH_CAPSULE')
        pie.operator(MakeRigidBodyConstraints.bl_idname, icon='LINKED')
        pie.operator(SetRBCSettings.bl_idname, icon='CONSTRAINT')
        pie.operator(UnlinkHitBoxAndBone.bl_idname, icon='LOCKVIEW_OFF')
        pie.operator(LinkHitBoxAndBone.bl_idname, icon='LOCKVIEW_ON')
        pie.operator(BakeToBone.bl_idname, icon='KEY_HLT')
        pie.operator(SetRBSettings.bl_idname, icon='MESH_ICOSPHERE')


class RagdollPieMenuStarter(bpy.types.Operator):
    """Ragdoll tools pie menu"""
    bl_idname = "ragdoll.pie_menu_start"
    bl_label = "Ragdoll tools pie menu"

    def invoke(self, context, event):
        bpy.ops.wm.call_menu_pie(name="RagdollPieMenu")
        return {'FINISHED'}


class_list = (BakeToBone, GenerateHitBoxes)


def register():
    # os.system('cls')
    # print("I AM REGISTER")

    register_extensions()
    bpy.utils.register_module(__name__)
    # print("REGISTER FINISHED")


def unregister():
    # print("I AM UNREGISTER")
    bpy.utils.unregister_module(__name__)
    # print("UNREGISTER FINISHED")


